/* Bump mapping shader by daxnitro.  
   This shader implements parallax occlusion and specular mapping.  It requires terrain_nh.png and terrain_s.png to be included in the current texture pack. */

// To disable world curvature, add two forward slashes to the beginning of the following line:

#define ENABLE_WORLD_CURVATURE

// !!!!!!!! THE TYPICAL USER DOESN'T NEED TO LOOK AT ANYTHING BELOW HERE !!!!!!!!

#version 120 // This will always get moved to the top of the code in pre-processing.

varying vec4 vertColor;

#ifdef _ENABLE_GL_TEXTURE_2D
centroid varying vec4 texCoord;

#ifdef _ENABLE_BUMP_MAPPING

varying vec3 viewVector;
varying vec3 lightVector;

uniform vec3 sunPosition;
uniform vec3 moonPosition;

varying vec4 specMultiplier;

uniform int worldTime;

varying float useCelestialSpecularMapping;

varying float distance;
#endif
#endif

#ifdef ENABLE_WORLD_CURVATURE
	const float WORLD_RADIUS         = 2500.0;
	const float WORLD_RADIUS_SQUARED = 6250000.0;
#endif

void main() {
	
#ifdef ENABLE_WORLD_CURVATURE
	
	vec4 position = gl_ModelViewMatrix * gl_Vertex;
	if (gl_Color.a != 0.8) {
		// Not a cloud.
		float flatDistanceSquared = position.x * position.x + position.z * position.z;
		position.y -= WORLD_RADIUS - sqrt(max(1.0 - flatDistanceSquared / WORLD_RADIUS_SQUARED, 0.0)) * WORLD_RADIUS;
#ifdef _ENABLE_BUMP_MAPPING
		distance = sqrt(flatDistanceSquared + position.y * position.y);
#endif
	}		
	gl_Position = gl_ProjectionMatrix * position;

#elif defined(_ENABLE_BUMP_MAPPING)

	vec4 position = gl_ModelViewMatrix * gl_Vertex;
	distance = sqrt(position.x * position.x + position.y * position.y + position.z * position.z);
	gl_Position = gl_ProjectionMatrix * position;

#else

	gl_Position = ftransform();
	
#endif

	vertColor = gl_Color;

#ifdef _ENABLE_GL_TEXTURE_2D

	texCoord = gl_MultiTexCoord0;

#ifdef _ENABLE_BUMP_MAPPING

	vec3 normal = normalize(gl_NormalMatrix * gl_Normal);
	vec3 tangent;
	vec3 binormal;
	
	useCelestialSpecularMapping = 1.0;

	if (gl_Normal.x > 0.5) {
		//  1.0,  0.0,  0.0
		tangent  = normalize(gl_NormalMatrix * vec3( 0.0,  0.0, -1.0));
		binormal = normalize(gl_NormalMatrix * vec3( 0.0, -1.0,  0.0));
	} else if (gl_Normal.x < -0.5) {
		// -1.0,  0.0,  0.0
		tangent  = normalize(gl_NormalMatrix * vec3( 0.0,  0.0,  1.0));
		binormal = normalize(gl_NormalMatrix * vec3( 0.0, -1.0,  0.0));
	} else if (gl_Normal.y > 0.5) {
		//  0.0,  1.0,  0.0
		tangent  = normalize(gl_NormalMatrix * vec3( 1.0,  0.0,  0.0));
		binormal = normalize(gl_NormalMatrix * vec3( 0.0,  0.0,  1.0));
	} else if (gl_Normal.y < -0.5) {
		//  0.0, -1.0,  0.0
		useCelestialSpecularMapping = 0.0;
		tangent  = normalize(gl_NormalMatrix * vec3( 1.0,  0.0,  0.0));
		binormal = normalize(gl_NormalMatrix * vec3( 0.0,  0.0,  1.0));
	} else if (gl_Normal.z > 0.5) {
		//  0.0,  0.0,  1.0
		tangent  = normalize(gl_NormalMatrix * vec3( 1.0,  0.0,  0.0));
		binormal = normalize(gl_NormalMatrix * vec3( 0.0, -1.0,  0.0));
	} else if (gl_Normal.z < -0.5) {
		//  0.0,  0.0, -1.0
		tangent  = normalize(gl_NormalMatrix * vec3(-1.0,  0.0,  0.0));
		binormal = normalize(gl_NormalMatrix * vec3( 0.0, -1.0,  0.0));
	}
	
	mat3 tbnMatrix = mat3(tangent.x, binormal.x, normal.x,
                          tangent.y, binormal.y, normal.y,
                          tangent.z, binormal.z, normal.z);
	
	viewVector = (gl_ModelViewMatrix * gl_Vertex).xyz;
	viewVector = normalize(tbnMatrix * viewVector);

	if (worldTime < 12000 || worldTime > 23250) {
		lightVector = normalize(tbnMatrix * -sunPosition);
		specMultiplier = vec4(1.0, 1.0, 1.0, 1.0);
	} else {
		lightVector = normalize(tbnMatrix * -moonPosition);
		specMultiplier = vec4(0.5, 0.5, 0.5, 0.5);
	}
	specMultiplier *= clamp(abs(float(worldTime) / 500.0 - 46.0), 0.0, 1.0) * clamp(abs(float(worldTime) / 500.0 - 24.5), 0.0, 1.0);

#endif // _ENABLE_GL_BUMP_MAPPING
#endif // _ENABLE_GL_TEXTURE_2D
	
	gl_FogFragCoord = gl_Position.z;
}